#
# NAME
#   mri.tcl
#
# DESCRIPTION
#   Mri Stimulator Definition File
#
# AUTHOR
#   DLS/REHBM
#

namespace eval ::mri {
    proc sys_init { } {
	set ::mri::stim_request {}
    }
    proc sys_destroy { } {
	# nothing to do
    }

    #######################################################################
    # default_stimcmds
    #   defines all the procedures that are called by the state system.
    #   each of these procedures can (and many should) be overwritten in
    #   the {protocol}.tcl file.
    #   these provide defaults for the state system dependent stimcmds that
    #   will be inherited by each protocol unless they are overwritten in
    #   {protocol}.tcl
    proc default_stimcmds { } {
	namespace inscope :: {
	    # showtrial <n>:
	    #    Description:
	    #       displays <trial> on stimulator
	    #       basically just sets the trial, loads the trial and
	    #       shows it, but can be more complicated if overwritten.
	    proc showtrial { trial } { 
		settrial $trial
		nextrun
		show 0
	    }

	    # settrial <n>:
	    #    Description:
	    #       sets the current stimdg:remaining list such that the
	    #       only valid trial is $n.  if $n is a number, then the trial
	    #       is set to that number.  if $n is 'next' then trial is 
	    #       incremented, if $n = 'prev' then trial is set to previous
	    #       trial.
	    #       trial is confined to 0<current(trial)<length stimdg
	    proc settrial { n } {
		switch $n {
		    next {
			incr current(trial)
		    }
		    prev {
			set current(trial) [expr $current(trial)-1]
		    }
		    default {
			set current(trial) $n
		    }	    
		}
		if {$current(trial) > [dl_length stimdg:remaining]} {
		    set current(trial) [dl_length stimdg:remaining]
		} elseif { $current(trial) < 0} {
		    set current(trial) 0
		}
		dl_set stimdg:remaining \
		    [dl_eq [dl_index stimdg:remaining] $current(trial)]
	    }


	    # reset:
	    #    Description:
	    #       clears the screen and resets the stimdg
	    proc reset { } {
		glistSetVisible 0; redraw;
		dl_set stimdg:remaining [dl_ones [dl_length stimdg:remaining]]
	    }

	    # fixon:
	    #    Description:
	    #       draws the fixspot
	    proc fixon {} {
		glistSetCurGroup 0; glistSetVisible 1; redraw;
	    }

	    # fixoff:
	    #    Description:
	    #       removes fixspot from screen (leaves it clear)
	    proc fixoff {} {
		glistSetVisible 0; redraw;
	    }
	    
	    # show <id>:
	    #    Description:
	    #    displays the main stimuli group
	    proc show { id } {
		glistSetCurGroup 1; glistSetVisible 1; redraw;
	    }

    
	    # clearscreen:
	    #    Description:
	    #       clears the screen
	    proc clearscreen { } {
		glistSetVisible 0; redraw;
	    }
	    
	    # nextrun:
	    #    Description:
	    #       chooses the upcomming trial's stim id form those remaining
	    #       loads all relevent information for the upcomming trial
	    #    Return:
	    #       if there are no more trials remaining.  block finished.
	    #          -1 $nstims $nstims
	    #       else   
	    #          id side class remaining_trials total_trials
	    proc nextrun {} {
		return "-1 0 0"
	    }
	    
	    # targinfo:
	    #    Description:
	    #       returns relevent information about the current target
	    #    Return:
	    #       x_coordinate y_coordinate scale
	    proc targinfo {} {
		return "0. 0. 0."
	    }


	    # fixate_in:
	    #    Description:
	    #       callback from QNX for entering fixwin
	    #    Return:
	    #       none
	    proc fixate_in {} { return }

	    # fixate_out:
	    #    Description:
	    #       callback from QNX for exitting fixwin
	    #    Return:
	    #       none
	    proc fixate_out {} { return }

	    
	    # targname:
	    #    Description:
	    #       returns name of current target
	    #    Return:
	    #       targ_name
	    proc targname {} {
		return null
	    }

	    # scenename:
	    #    Description:
	    #       returns name of current scene (background)
	    #    Return:
	    #       background_name
	    proc scenename {} {
		return blank
	    }
	    
	    # delayinfo:
	    #    Description:
	    #       returns delay after show is called that target appears
	    #    Return:
	    #       delay(msec)
	    proc delayinfo { } {
		return 0
	    }

	    # postfixtime:
	    #    Description:
	    #       returns delay between fixoff and stimon
	    #    Return:
	    #       delay(msec)
	    proc postfixtime { } {
		return -1
	    }

	    # rewardinfo:
	    #    Description:
	    #       returns reward duration
	    #    Return:
	    #       msec for reward
	    proc rewardinfo { } {
		return 0
	    }

	    # posttrialstimtime:
	    #    Description:
	    #       returns delay after endrun to show stim (feedback)
	    #    Return:
	    #       delay(msec)
	    proc posttrialstimtime { } {
		return 0
	    }

	    # stimulus_duration:
	    #    Description:
	    #       returns duration of stimulus presentation (or 0, which
	    #      means to leave for entire response period)
	    #    Return:
	    #       duration(msec) or 0
	    proc stimulus_duration { } {
		return 0
	    }

	    # fixwin:
	    #    Description:
	    #       Window size for maintained fixation
	    #    Return:
	    #       width height
	    proc fixwin { } {
		return "12. 10."
	    }

	    # stimulationinfo:
	    #    Description:
	    #       returns relevenat information regarding any cortical 
	    #       stimulation on the upcomming trial
	    #    Return:
	    #       stimulating? delay duration
	    proc stimulationinfo { } {
		return "0 0 0"
	    }
	    

	    # subjectinfo:
	    #    Description:
	    #       can override subject info string for data file naming
	    proc subjectinfo { subj } {
		return
	    }

	    # datafileinfo:
	    #    Description:
	    #       can override datafile info string for data file naming
	    proc datafileinfo { } {
		return
	    }

	    # estim_cmd:
	    #    Description:
	    #       allows stim computer to send a command to stimulator
	    #       possible commands are "on" and "off"
	    #    Return:
	    #       none
	    proc estim_cmd { cmd } {
	    }

	    # stim_request:
	    #    Description:
	    #       allows stim computer to send a request back to QNX
	    #    Return:
	    #       request string
	    proc stim_request {} {    
		return $::mri::stim_request
	    }

            # reward <duration>
            #    Description:
            #       be notified that remote system has delivered a reward 
            proc reward { duration } {
            }

	    # endrun <status>:
	    #    Description:
	    #       determine, given the status of trial, if it should be 
	    #       removed from remaining trials for the current block.
	    #       By default, the trial will be removed if it was 'wrong'
	    proc endrun { } {
	    }

	    # abort <no_response>:
	    #    Description:
	    #       determine what to do upon abort
	    proc abort { no_response } {
	    }

	    # closedata:
	    #    Description:
	    #       action to take when a datafile is closed (taken from io.tcl)
	    proc closedata {} {
		set ::datafile_open 0
	    }
	}
    }
    
    set rmt_cmds { 
	clearscreen
	closedata
	delayinfo
	endrun
	fixoff
	fixon
	nextrun
	{opendata {%s filename} {%d flags}}
	presession
	postrun
	reset
	scenename
	{show {%d stim}}
	targinfo
	targname
	test_proc
	trigger
	stimulationinfo
	{settrial {%d trial}}
	stim_request
    }
    
    set cooked_rmt_cmds {
	clearscreen
	delayinfo
	endrun
	fixoff
	fixon
	nextrun
	reset
	scenename
	{show {%d stim}}
	targinfo
	targname
	trigger
	stimulationinfo
	{ nextrun-show }
	{ settrial {%d trial}; nextrun; show 0 }
	stim_request
    }

}
